//------------------------------------------------------------------
//  TestMath.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeMath from "../../util/math";
import * as axeString from "../../util/string";
import { Point } from "../../util/Point";
import { UnitTest } from "../../unittest/UnitTest";
import { UnitTestRunner } from "../../unittest/UnitTestRunner";
//------------------------------------------------------------------

/** Unit test for the Math class.
 */
export class TestMath extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new Math object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /* Test the perpendicularPoint() function for a point in each of 4 quadrants.
   */
  testMathPerpendicularPointQuadrants() : void
  {
    /////////////////////////////////////////////////////
    // Quadrent 1.

    let line1 = new Point(1, 1);
    let line2 = new Point(3, 5);

    // Distance is (1, 2) away.
    let point = new Point(4, 7);

    // Clockwise point.

    let expected = new Point(5, 4);
    let actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Q1 clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(1, 6);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Q1 counter-clockwise", expected, actual);

    /////////////////////////////////////////////////////
    // Quadrent 2.

    line1 = new Point(1, 1);
    line2 = new Point(-3, 5);

    // Distance is (2, 2) away.
    point = new Point(-1, 3);

    // Clockwise point.

    expected = new Point(-1, 7);
    actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Q2 clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(-5, 3);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Q2 counter-clockwise", expected, actual);

    /////////////////////////////////////////////////////
    // Quadrent 3.

    line1 = new Point(1, 1);
    line2 = new Point(-1, -5);

    // Distance is (1, 3) away.
    point = new Point(0, -2);

    // Clockwise point.

    expected = new Point(-4, -4);
    actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Q3 clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(2, -6);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Q3 counter-clockwise", expected, actual);

    /////////////////////////////////////////////////////
    // Quadrent 4.

    line1 = new Point(-1, -2);
    line2 = new Point(19, -22);

    // Distance is (5, 5) away.
    point = new Point(24, -17);

    // Clockwise point.

    expected = new Point(14, -27);
    actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Q4 clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(24, -17);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Q4 counter-clockwise", expected, actual);
  }

  /* Test the perpendicularPoint() function for vertical/horizontal lines.
   */
  testMathPerpendicularPointVertical() : void
  {
    /////////////////////////////////////////////////////
    // Up

    let line1 = new Point(1, 1);
    let line2 = new Point(1, 2);

    // Distance is (0, 5) away.
    let point = new Point(1, 7);

    // Clockwise point.

    let expected = new Point(6, 2);
    let actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Up clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(-4, 2);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Up counter-clockwise", expected, actual);

    /////////////////////////////////////////////////////
    // Left

    line1 = new Point(1, 1);
    line2 = new Point(0, 1);

    // Distance is (0, 5) away.
    point = new Point(0, 6);

    // Clockwise point.

    expected = new Point(0, 6);
    actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Left clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(0, -4);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Left counter-clockwise", expected, actual);

    /////////////////////////////////////////////////////
    // Down

    line1 = new Point(1, 1);
    line2 = new Point(1, 0);

    // Distance is (0, 5) away.
    point = new Point(1, 5);

    // Clockwise point.

    expected = new Point(-4, 0);
    actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Down clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(6, 0);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Down counter-clockwise", expected, actual);

    /////////////////////////////////////////////////////
    // Right

    line1 = new Point(0, 0);
    line2 = new Point(10, 0);

    // Distance is (0, 5) away.
    point = new Point(5, 0);

    // Clockwise point.

    expected = new Point(10, -5);
    actual = axeMath.perpendicularPoint(line1, line2, point, true);

    this.assertPointClose("Right clockwise", expected, actual);

    // Counter-clockwise point.

    expected = new Point(10, 5);
    actual = axeMath.perpendicularPoint(line1, line2, point, false);

    this.assertPointClose("Right counter-clockwise", expected, actual);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

  private assertPointClose(msg: string, expected: Point, actual: Point) : void
  {
    this.assertClose(msg + " x", expected.x, actual.x);
    this.assertClose(msg + " y", expected.y, actual.y);
  }

} // END class TestMath

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestMath());
