//------------------------------------------------------------------
//  func.ts
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

//------------------------------------------------------------------
import { AnyFunction } from './types';
import * as axeString from "../util/string";
//------------------------------------------------------------------

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Returns a bound method, a function that calls one object's method.
 *
 * This is useful to be able to use a method where a function is expected.
 *
 * For example:
 *
 *   var myObj = new MyClass();
 *   myObj.doSomething();
 *
 *   var myFunc = axe.function.bind(myObj, myObj.doSomething);
 *
 *   myFunc();  // Same as calling myObj.doSomething();
 *
 *   // You can pass in a string instead of a function object.
 *
 *   var myFunc = axe.function.bind(myObj, 'doSomething');
 *
 * This is a varargs function.  Any extra arguments beyond 'obj' and 'method'
 * will be passed into the method call.
 *
 * @param obj The object whose method will be called.
 * @param method The method on the passed in object.  This may be either
 *                a function object or the string name of the function.
 *
 * @return A new function that is the bound method.
 */
export function bind(obj: any, method: string, ...args: any[]) : AnyFunction
{
  let methodObj: AnyFunction = null;

  if (typeof method == 'string')
  {
    methodObj = getMethod(obj, method);
  }
  else if (typeof method != 'function')
  {
    let msg = ("Error binding method.  The passed-in method " +
               "should be a function, but instead its type is: " +
               typeof method);
    throw new Error(msg);
  }
  else
  {
    methodObj = <AnyFunction><any>method;
  }

  return bind2.apply(null, [obj, methodObj].concat(args));
}

export function bind2<T extends AnyFunction>(obj: any,
                                             method: T,
                                             ...args: any[]) : T
{
  let boundArguments = args;

  // Create a closure with the object, method, and arguments.
  let newFunc: AnyFunction = function(...calledArguments: any[]) : any
  {
    let newArgs: Array<any> = [];
    for (let i = 0; i < boundArguments.length; ++i)
    {
      newArgs.push(boundArguments[i]);
    }
    for (let i = 0; i < calledArguments.length; ++i)
    {
      newArgs.push(calledArguments[i]);
    }

    return method.apply(obj, newArgs);
  };

  // Add properties to the function so you can retrieve the original
  // object and method.
  (<any>newFunc).createdByFuncBind = true;
  (<any>newFunc).boundObj = obj;
  (<any>newFunc).boundMethod = method;

  // Unfortunately, we need to cast to T because we cannot declare 'newFunc'
  // to accept and return the same types as function T.
  return <T>newFunc;
}

/** Returns a function that returns the specified object.
 *
 * This function can be useful when you have an object, but you're
 * using an API that wants a function that returns an object.
 *
 * @param object The object that will be returned by the returned function.
 *
 * @returns A function.
 */
export function returns<T>(object: T) : () => T
{
  let newFunc = function() : T
  {
    return object;
  };

  return newFunc;
}

/** Returns a function wrapped an exception-safe wrapper.
 *
 * This is useful to wrap a callback being passed to a framework
 * which is swallowing exceptions.
 */
export function exceptionSafe<T extends AnyFunction>(baseFunc: T) : T
{
  let func: AnyFunction = function(...args: any[]) : any
  {
    try
    {
      return baseFunc.apply(this, args);
    }
    catch (ex)
    {
      console.error(ex);
      throw ex;
    }
  };

  return <T>func;
}


/** Returns true if the two functions are the same.
 *
 * The two function are the same if they are both the same function object,
 * or if they are bound methods that are call the same object's method.
 */
export function areSame(func1: AnyFunction, func2: AnyFunction) : boolean
{
  let func1any = <any>func1;
  let func2any = <any>func2;

  if (!func1 && !func2)
  {
    return true;
  }
  else if (func1 === func2)
  {
    return true;
  }
  else if ((func1any.createdByFuncBind && func2any.createdByFuncBind) &&
           (func1any.boundObj === func2any.boundObj) &&
           (func1any.boundMethod === func2any.boundMethod))
  {
    return true;
  }
  else
  {
    return false;
  }
}

/** Slices function arguments in a browser-safe way.
 *
 * IE 7 doesn't allow you to call 'slice' on its 'arguments' object.
 */
export function slice(theArguments: IArguments, startIndex: number) : Array<any>
{
  if (startIndex >= theArguments.length) {
    return [];
  }

  let ret: Array<any> = [];
  for (let i = startIndex; i < theArguments.length; ++i) {
    ret.push(theArguments[i]);
  }
  return ret;
}

/** Given and object and method name, returns the function object.
 *
 * If no method with that name exists, throws an exception.
 */
export function getMethod(object: any, functionName: string) : AnyFunction
{
  if (typeof object[functionName] == 'undefined')
  {
    let msg = ("Error getting method '" + functionName + "'.  No " +
               "such method exists on object: " + object);
    throw new Error(msg);
  }

  let method = object[functionName];
  if (typeof method != 'function')
  {
    let msg = ("Error getting method '" + functionName + "'.  The property " +
               "should be a function, but instead its type is: " +
               typeof method);
    throw new Error(msg);
  }

  return method;
}

/** Returns a getter function that returns the specified property of an object.
 *
 *  This getter:
 *
 *    let getter = axe.func.getter('foo');
 *    let fooValue = getter(obj);
 *
 *  is equivalent to:
 *
 *    let fooValue = obj.foo;
 */
export function getter<T>(propertyName: string) : (obj: any) => T
{
  let func = function(theObject: any) : T
  {
    return theObject[propertyName];
  };

  return func;
}

/** Returns a setter function that sets the specified property of an object.
 *
 * The value to set may be passed in once and used for every call:
 *
 *   let setter = axe.func.setter('foo', 3);
 *   setter(obj);
 *
 * Or the value may be passed into each call:
 *
 *   let setter = axe.func.setter('foo');
 *   setter(obj, 3);
 *
 * Both are equivalent to:
 *
 *   obj.foo = 3;
 */
export function setter<T>(propertyName: string,
                          value?: T) : (obj: any, value?: T) => void
{
  let hasConstValue = (typeof value != 'undefined');
  let constValue = hasConstValue ? value : null;

  let func = function(theObject: any, valueParameter?: T) : void
  {
    let hasFunctionValue = (typeof valueParameter != 'undefined');
    let functionValue = hasFunctionValue ? valueParameter : null;

    if (!hasConstValue && !hasFunctionValue)
    {
      throw Error("Called setter with no value to set.");
    }

    let valueToSet = functionValue;
    if (hasConstValue)
    {
      valueToSet = constValue;
    }

    theObject[propertyName] = valueToSet;
  };

  return func;
}

/** Creates a caller function that calls the specified method of an object.
 *
 * The returned function will call the specified method on any object
 * that it is called with.
 *
 * Any args beyond the method name will be passed as argments.  For example:
 *
 *    let myFunc = axe.func.caller('foo', 1, 2, 3);
 *    myFunc(obj);
 *
 * or:
 *
 *    let myFunc = axe.func.caller('foo');
 *    myFunc(obj, 1, 2, 3);
 *
 * is equivalent to:
 *
 *    obj.foo(1, 2, 3);
 *
 * @param methodName The method to be called on each object.
 */
export function caller(methodName: string, ...methodArgs: any[]) : AnyFunction
{
  let func = function(theObject: any) : any
  {
    let boundMethod = bind(theObject, methodName);

    let calledArgs = slice(arguments, 1);
    let allArgs = [].concat(methodArgs).concat(calledArgs);

    // Call the bound method with no 'this' and the specified args.
    return boundMethod.apply(null, allArgs);
  };

  return func;
}

/** Wraps a single-arg function to take two args.
 *
 * In the returned function, the first arg is ignored, and the
 * second arg is passed to the wrapped function.
 *
 * This is useful for D3 callbacks where the first argument is null and
 * and second argument is the selection index.
 */
export
function wrap21<T, U>(wrappedFunc: (arg: T) => U) : (obj: any, arg: T) => U
{

  let func = function(dummy: any, arg: T) : U
  {
    return wrappedFunc(arg);
  };

  return func;
}

