//------------------------------------------------------------------
//  classes.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import { AnyCtor } from './types';
import { AnyFunction } from './types';
import { contains } from './array';
//------------------------------------------------------------------

//------------------------------------------------------------------
// Constants
//------------------------------------------------------------------

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Returns the name of the class of which the specified object is an instance.
 *
 * For example:
 *
 *  class A {};
 *  var a = new A();
 *
 *  className(a); => returns the string 'A'
 */
export function className(obj: any) : string
{
  if (obj === null)
  {
    return 'null';
  }
  else if (obj && typeof obj == 'object' && ('constructor' in obj))
  {
    return obj.constructor.name;
  }
  else
  {
    return typeof obj;
  }
}

/** Returns a list of all methods on an object.
 */
export function methodNames(obj: object) : Array<string>
{
  let propNames: Array<string> = [];

  let theObj = obj;
  while (theObj)
  {
    propNames = propNames.concat(Object.getOwnPropertyNames(theObj));
    theObj = Object.getPrototypeOf(theObj);
  }

  function isNotDuplicate(item: string,
                          index: number,
                          array: Array<string>) : boolean
  {
    return item != array[index + 1];
  }

  function isFunction(item: string) : boolean
  {
    return typeof (<any>obj)[item] == 'function';
  }

  propNames.sort();
  propNames = propNames.filter(isNotDuplicate);
  propNames = propNames.filter(isFunction);

  return propNames;
}

/** True if a class inherits from another.
 *
 * For example if you have classes:
 *
 *   class A  {}
 *   class B extends A {}
 *   class C extends B {}
 *   class D extends C {}
 *   class Z {}
 *
 * then calling isBaseClass() will have the following results:
 *
 *   isBaseClass(A, B) => true
 *   isBaseClass(B, A) => false
 *   isBaseClass(A, D) => true
 *   isBaseClass(A, Z) => false
 *
 * @param baseClass The class to test if it is the base.
 * @param derivedClass The class to test if it is derived from base.
 *
 * @return True if the classes are related by inheritance.
 *         False if they are not.
 */
export function isBaseClass(baseClass: AnyCtor, derivedClass: AnyCtor) : boolean
{
  return contains(baseClasses(derivedClass), baseClass);
}

/** Returns a list of superclasses for the specified class.
 *
 * For example if you have classes:
 *
 *   class A  {}
 *   class B extends A {}
 *   class C extends B {}
 *   class D extends C {}
 *
 * then calling:
 *
 *   baseClasses(D)
 *
 * will return:
 *
 *   [C, B, A]
 *
 * @param subClass A subclass object.
 *
 * @return A list of superclass objects.
 */
export function baseClasses(clazz: AnyCtor) : Array<AnyCtor>
{
  // This implementation seems consistent with how Typescript creates
  // classes, but was created via trial-and-error.  It may need to be fixed.

  // Multiple inheritance was *not* considered and is not supported.

  let ret: Array<AnyCtor> = [];

  // The prototype for the current level of inheritance.
  let proto: any = clazz.prototype.__proto__;

  while (proto.constructor != Object)
  {
    ret.push(proto.constructor);
    proto = proto.__proto__;
  }

  return ret;
}
