//------------------------------------------------------------------
//  MapSetterValue.ts
//  Copyright 2014 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import { Map } from './Map';

/** Helper class for Map to allow property get/set access to values.
 *
 * This class is mostly to allow AngularJS ng-model to be used
 * with a Map value.
 *
 * For example, to tell an Angular <input> to get/set the value
 * for the key 3, you would do:
 *
 *   ng-model='myMap.access(3).value'
 *
 * myMap.access(3) returns a MapSetterValue that get/sets the
 * value for the 3 key if you get/set its value property.
 */
export class MapSetterValue<T, U>
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The map that values will be get/set on.
   */
  theMap: Map<T, U>;

  /** The key that will be get/set.
   */
  key: T;

  //----------------------------------------------------------------
  // Initialization
  //----------------------------------------------------------------

  /** Initializes a new MapSetterValue object.
   */
  constructor(theMap: Map<T, U>, key: T)
  {
    this.theMap = theMap;
    this.key = key;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Sets the value for this object's map and key.
   *
   * This property function is called when you assign to the 'value' property.
   *
   * @param value The new value to set.
   */
  set value(value: U)
  {
    this.theMap.put(this.key, value);
  }

  /** Gets the value for this object's map and key.
   *
   * This property function is called when you read from the 'value' property.
   *
   * @return The value for this object's map and key.
   */
  get value() : U
  {
    return this.theMap.get(this.key);
  }

} // END class MapSetterValue
