//------------------------------------------------------------------
//  Day.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeDate from "../util/date";
import * as axeString from "../util/string";
//------------------------------------------------------------------

/** Add class documentation here.
 */
export class Day
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The number of milliseconds after the Date epoch.
   */
  private dateMillis: number;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new Day from a year, month, and day.
   *
   * The year, month and day arguments are treated just like
   * the Date constructor treats its arguments.  Note that
   * year and days are zero-based but months are one-based!
   *
   * @param year The year number.
   * @param month The month number (1-based).
   * @param day The day of the month number.
   */
  constructor(year: number, month: number, day: number);

  /** Creates a new Day from a number of milliseconds since the epoch.
   *
   * @param datetime The number of milliseconds after the Date epoch.
   */
  constructor(datetime: number);

  /** Constructor implementation.
   */
  constructor(yearOrMillis: number, month?: number, day?: number)
  {
    // If the constructor was called with only one argument, it's a number
    // of milliseconds since the epoch.
    let dateMillis: number = yearOrMillis;

    // If the constructor was called with multiple arguments, they are
    // year, month, and day.
    if (typeof month != 'undefined')
    {
      let theDate = new Date(yearOrMillis, month, day);
      dateMillis = theDate.getTime();
    }

    this.dateMillis = dateMillis;
  }

  /** Creates a day for the current day.
   */
  static today() : Day
  {
    let today = new Day(Day.normalize(Date.now()));
    return today;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Normalize a date by throwing away any excess hours, minutes, seconds.
   */
  static normalize(dateTimeMillis: number) : number
  {
    let theDate = new Date(dateTimeMillis);
    axeDate.discardDayFraction(theDate);

    return theDate.getTime();
  }

  /** Returns this day's number of seconds past 1970.
   */
  valueOf() : number
  {
    return this.dateMillis;
  }

  /** Returns this day's number of seconds past 1970.
   */
  asDate() : Date
  {
    return new Date(this.dateMillis);
  }

  /** Returns whether the 2 objects are equal.
   */
  equals(other: Day) : boolean
  {
    if (other == null) {
      return false;
    }

    return this.valueOf() == other.valueOf();
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    return axeDate.formatDate(this);
  }

} // END class Day
