//------------------------------------------------------------------
//  TestClassJsonDecoder.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../util/string";
import { ObjectMap } from "../../util/types";
import { Map } from "../../util/Map";
import { ClassJsonDecoder } from '../ClassJsonDecoder';
import { ClassJsonDesc } from '../ClassJsonDesc';
import { ClassJsonRegistry } from '../ClassJsonRegistry';
import { Day } from "../../date/Day";
import { Duration } from "../../date/Duration";
import { JsonDate } from "../jsonTypes/JsonDate";
import { JsonDateTime } from "../jsonTypes/JsonDateTime";
import { JsonDuration } from "../jsonTypes/JsonDuration";
import { JsonLink } from "../jsonTypes/JsonLink";
import { JsonList } from "../jsonTypes/JsonList";
import { JsonMap } from "../jsonTypes/JsonMap";
import { JsonObj } from "../jsonTypes/JsonObj";
import { JsonPrimitiveType } from "../jsonTypes/JsonPrimitiveType";
import { UnitTest } from '../../unittest/UnitTest';
import { UnitTestRunner } from '../../unittest/UnitTestRunner';
//------------------------------------------------------------------

/** Unit test for the ClassJsonDecoder class.
 */
export class TestClassJsonDecoder extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ClassJsonDecoder object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods
  //------------------------------------------------------------------

  /** Test decoding psuedo-primitive types.
   */
  testDecodeBuiltins() : void
  {
    interface IDateHolder
    {
      date: Day;
      datetime: Date;
      duration: Duration;
      dateList: Array<Day>;
      datetimeList: Array<Date>;
      durationList: Array<Duration>;
      dateMap: Map<string, Day>;
      datetimeMap: Map<string, Date>;
      durationMap: Map<string, Duration>;
    }

    class DateHolder
    {
      dateMap: Map<string, Day> = Map.createStringMap<Day>();
      datetimeMap: Map<string, Date> = Map.createStringMap<Date>();
      durationMap: Map<string, Duration> = Map.createStringMap<Duration>();

      /* tslint:disable:no-parameter-properties */
      constructor(public date: Day,
                  public datetime: Date,
                  public duration: Duration,
                  public dateList: Array<Day>,
                  public datetimeList: Array<Date>,
                  public durationList: Array<Duration>)
      {
      }

      static fromJson(src: IDateHolder) : DateHolder
      {
        let obj = new DateHolder(src.date, src.datetime, src.duration,
                                 src.dateList, src.datetimeList,
                                 src.durationList);
        obj.dateMap.putAll(src.dateMap);
        obj.datetimeMap.putAll(src.datetimeMap);
        obj.durationMap.putAll(src.durationMap);

        return obj;
      }

      static toJson(src: DateHolder) : IDateHolder
      {
        return src;
      }
    }

    let dateHolderDesc = new ClassJsonDesc("DateHolder");

    dateHolderDesc.addField("date", new JsonDate());
    dateHolderDesc.addField("datetime", new JsonDateTime());
    dateHolderDesc.addField("duration", new JsonDuration());
    dateHolderDesc.addField("dateList", new JsonList(new JsonDate()));
    dateHolderDesc.addField("datetimeList", new JsonList(new JsonDateTime()));
    dateHolderDesc.addField("durationList", new JsonList(new JsonDuration()));
    dateHolderDesc.addField("dateMap",
                            new JsonMap(new JsonPrimitiveType("string"),
                                        new JsonDate()));
    dateHolderDesc.addField("datetimeMap",
                            new JsonMap(new JsonPrimitiveType("string"),
                                        new JsonDateTime()));
    dateHolderDesc.addField("durationMap",
                            new JsonMap(new JsonPrimitiveType("string"),
                                        new JsonDuration()));

    ClassJsonRegistry.registry.addDesc(dateHolderDesc);
    ClassJsonRegistry.registry.register("DateHolder", DateHolder);

    // 2016-01-02
    let theDate = new Day(2016, 0, 2);

    // 2016-01-04T01:02:03.456Z
    let theDatetime = new Date(Date.UTC(2016, 0, 4, 1, 2, 3, 456));

    let theDuration = new Duration(1000);

    let dateHolder = new DateHolder(theDate,
                                    theDatetime,
                                    theDuration,
                                    [theDate],
                                    [theDatetime],
                                    [theDuration]);
    dateHolder.dateMap.put("date", theDate);
    dateHolder.datetimeMap.put("datetime", theDatetime);
    dateHolder.durationMap.put("duration", theDuration);

    let jsonStr: string = `{
      "_class": "DateHolder",
      "date": "2016-01-02",
      "datetime": "2016-01-04T01:02:03.456Z",
      "duration": 1000,
      "dateList": ["2016-01-02"],
      "datetimeList": ["2016-01-04T01:02:03.456Z"],
      "durationList": [1000],
      "dateMap" : [
        ["date", "2016-01-02"]
      ],
      "datetimeMap" : [
        ["datetime", "2016-01-04T01:02:03.456Z"]
      ],
      "durationMap" : [
        ["duration", 1000]
      ]
    }`;

    let expected = dateHolder;
    let actual = (new ClassJsonDecoder()).decode(jsonStr);

    this.assertEqual("dates", expected, actual);
  }

  /** Test classes.
   */
  testEncodeNestedClasses() : void
  {
    let wheelDesc = new ClassJsonDesc("Wheel");
    wheelDesc.addField("diameter", new JsonPrimitiveType("int"));
    wheelDesc.addField("pressure", new JsonPrimitiveType("float"));

    let carDesc = new ClassJsonDesc("Car");
    carDesc.addField("wheels", new JsonList(new JsonObj("Wheel")));
    carDesc.addField("namedWheels",
                     new JsonMap(new JsonPrimitiveType("string"),
                                 new JsonObj("Wheel")));
    carDesc.addField("color", new JsonPrimitiveType("string"));

    ClassJsonRegistry.registry.addDesc(wheelDesc);
    ClassJsonRegistry.registry.addDesc(carDesc);

    interface ICar
    {
      color: string;
      wheels: Array<Wheel>;
      namedWheels: Map<string, Wheel>;
    }

    class Car implements ICar
    {
      namedWheels: Map<string, Wheel> = Map.createStringMap<Wheel>();

      constructor(public color: string, public wheels: Array<Wheel>)
      {
      }

      static fromJson(src: ICar) : Car
      {
        let newCar: Car = new Car(src.color, src.wheels);
        newCar.namedWheels = src.namedWheels;
        return newCar;
      }

      static toJson(src: Car) : ICar
      {
        return src;
      }
    }

    interface IWheel
    {
      diameter: number;
      pressure: number;
    }

    class Wheel implements IWheel
    {
      constructor(public diameter: number, public pressure: number)
      {
      }

      static fromJson(src: IWheel) : Wheel
      {
        return new Wheel(src.diameter, src.pressure);
      }

      static toJson(src: Wheel) : IWheel
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register("Car", Car);
    ClassJsonRegistry.registry.register("Wheel", Wheel);

    let jsonStr: string = `{
      "_class" : "Car",
      "color" : "red",
      "namedWheels" : [
        ["front", {"_class": "Wheel", "diameter": 3, "pressure": 3.14}],
        ["back", {"_class": "Wheel", "diameter": 4, "pressure": 3.15}]
      ],
      "wheels" : [
        {"_class": "Wheel", "diameter": 5, "pressure": 3.16},
        {"_class": "Wheel", "diameter": 6, "pressure": 3.17}
      ]
    }`;

    let obj = (new ClassJsonDecoder()).decode(jsonStr);

    this.assertEqual("attribute", "red", obj.color);
    this.assertEqual("wheel1", 5, obj.wheels[0].diameter);
    this.assertEqual("wheel2", 6, obj.wheels[1].diameter);
    this.assertEqual("front", 3, obj.namedWheels.get("front").diameter);
    this.assertEqual("back", 4, obj.namedWheels.get("back").diameter);
  }

  /** An unknown class should throw an exception.
   */
  testUnknownClass() : void
  {
    let jsonStr = '{"_class": "bad_class_name"}';

    try
    {
      (new ClassJsonDecoder()).decode(jsonStr);
      this.fail("Failed to throw unknown class exception");
    }
    catch (ex)
    {
    }
  }

  /** A extra property should throw an exception.
   */
  testExtraProperty() : void
  {
    let desc = new ClassJsonDesc("NamedPoint");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("x", new JsonPrimitiveType("int"));
    desc.addField("y", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedPoint
    {
      constructor(public name: string, public x: number, public y: number)
      {
      }

      static fromJson(src: ObjectMap<any>) : NamedPoint
      {
        return new NamedPoint(<string>src['name'],
                              <number>src['x'],
                              <number>src['y']);
      }

      static toJson(src: NamedPoint) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedPoint);


    let jsonStr = ('{"_class": "NamedPoint", ' +
                   '"name": "a name", ' +
                   '"x": 45, ' +
                   '"y": 46,' +
                   '"extraProp": "blah"}');

    try
    {
      (new ClassJsonDecoder()).decode(jsonStr);
      this.fail("Failed to throw missing property exception");
    }
    catch (ex)
    {
    }
  }

  /** A missing property should throw an exception.
   */
  testMissingProperty() : void
  {
    let desc = new ClassJsonDesc("NamedPoint");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("x", new JsonPrimitiveType("int"));
    desc.addField("y", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedPoint
    {
      constructor(public name: string, public x: number, public y: number)
      {
      }

      static fromJson(src: ObjectMap<any>) : NamedPoint
      {
        return new NamedPoint(<string>src['name'],
                              <number>src['x'],
                              <number>src['y']);
      }

      static toJson(src: NamedPoint) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedPoint);


    let jsonStr = '{"_class": "NamedPoint", "name": "a name", "x": 45}';

    try
    {
      (new ClassJsonDecoder()).decode(jsonStr);
      this.fail("Failed to throw missing property exception");
    }
    catch (ex)
    {
    }
  }

  /** Test an exception is thrown for a bad field type.
   */
  testBadType() : void
  {
    let desc = new ClassJsonDesc("NamedPoint");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("x", new JsonPrimitiveType("int"));
    desc.addField("y", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedPoint
    {
      constructor(public name: string, public x: number, public y: number)
      {
      }

      static fromJson(src: ObjectMap<any>) : NamedPoint
      {
        return new NamedPoint(<string>src['name'],
                              <number>src['x'],
                              <number>src['y']);
      }

      static toJson(src: NamedPoint) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedPoint);


    // Make y a string instead of an int.
    let jsonStr =
      '{"_class": "TestClass4", "name": "hello", "x": "3", "y": "bad string!"}';

    try
    {
      (new ClassJsonDecoder()).decode(jsonStr);
      this.fail("Failed to throw a bad field type exception.");
    }
    catch (ex)
    {
    }
  }

  /** Test JSON decoding of nested dictionaries.
   */
  testNestedDictDecode() : void
  {
    let desc = new ClassJsonDesc("DictHolder");
    desc.addField("theDict",
                  new JsonMap(new JsonPrimitiveType("int"),
                              new JsonMap(new JsonPrimitiveType("int"),
                                          new JsonPrimitiveType("bool"))));
    ClassJsonRegistry.registry.addDesc(desc);

    class DictHolder
    {
      constructor(public theDict: Map<number, Map<number, boolean>>)
      {
      }

      static fromJson(src: ObjectMap<any>) : DictHolder
      {
        let parsedDict = <Map<number, Map<number, boolean>>>src['theDict'];
        return new DictHolder(parsedDict);
      }

      static toJson(src: DictHolder) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, DictHolder);


    let subDict1 = Map.createIntMap<boolean>();
    let subDict2 = Map.createIntMap<boolean>();
    let expectedDict = Map.createIntMap<Map<number, boolean>>();

    subDict1.putPairs([[1, true], [2, true], [3, false]]);
    subDict2.putPairs([[4, true], [5, true], [6, false]]);
    expectedDict.putPairs([[100, subDict1], [101, subDict2]]);

    let encodedSubDict1 = '[[1, true], [2, true], [3, false]]';
    let encodedSubDict2 = '[[4, true], [5, true], [6, false]]';
    let encodedDict = ('[[100, ' + encodedSubDict1 + '], ' +
                       '[101, ' + encodedSubDict2 + ']]');
    let encodedObj = ('{"_class": "DictHolder", ' +
                      '"theDict": ' + encodedDict + '}');

    let actualObj = (new ClassJsonDecoder()).decode(encodedObj);
    this.assertEqual('nested dict', expectedDict, actualObj.theDict);
  }

  /** Test JSON encoding of nested lists.
   */
  testNestedListDecode() : void
  {
    let intType = new JsonPrimitiveType("int");

    let desc = new ClassJsonDesc("ListHolder");
    desc.addField("theList",
                  new JsonList(new JsonList(new JsonList(intType))));
    ClassJsonRegistry.registry.addDesc(desc);

    class ListHolder
    {
      constructor(public theList: Array<Array<Array<number>>>)
      {
      }

      static fromJson(src: ObjectMap<any>) : ListHolder
      {
        return new ListHolder(<Array<Array<Array<number>>>>src['theList']);
      }

      static toJson(src: ListHolder) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, ListHolder);


    let expectedList = [
      [[0, 1], [0, 2], [0, 3]],
      [[1, 1], [1, 2], [1, 3]],
    ];

    let jsonStr = ('{"_class": "ListHolder", ' +
                   '"theList":  [ [[0, 1], [0, 2], [0, 3]],' +
                   '[[1, 1], [1, 2], [1, 3]]]}');


    let actual = (new ClassJsonDecoder()).decode(jsonStr);

    this.assertEqual('decoded list', expectedList, actual.theList);
  }


  /** Test JSON decoding of a subclass.
   */
  testSubclass() : void
  {
    // A base class (NamedThing) and two subclasses (NamedPoint, NamedAmount).

    let desc = new ClassJsonDesc("NamedThing");
    desc.setAbstract(true);
    desc.addField("name", new JsonPrimitiveType("string"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedThing
    {
      constructor(public name: string)
      {
      }

      static fromJson(src: ObjectMap<any>) : NamedThing
      {
        return new NamedThing(<string>src['name']);
      }

      static toJson(src: NamedThing) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedThing);


    desc = new ClassJsonDesc("NamedPoint");
    desc.setBaseClass("NamedThing");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("x", new JsonPrimitiveType("int"));
    desc.addField("y", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedPoint extends NamedThing
    {
      constructor(name: string, public x: number, public y: number)
      {
        super(name);
      }

      static fromJson(src: ObjectMap<any>) : NamedPoint
      {
        return new NamedPoint(<string>src['name'],
                              <number>src['x'],
                              <number>src['y']);
      }

      static toJson(src: NamedPoint) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedPoint);


    desc = new ClassJsonDesc("NamedAmount");
    desc.setBaseClass("NamedThing");
    desc.addField("name", new JsonPrimitiveType("string"));
    desc.addField("amount", new JsonPrimitiveType("int"));
    ClassJsonRegistry.registry.addDesc(desc);

    class NamedAmount extends NamedThing
    {
      constructor(name: string, public amount: number)
      {
        super(name);
      }

      static fromJson(src: ObjectMap<any>) : NamedAmount
      {
        return new NamedAmount(<string>src['name'],
                               <number>src['amount']);
      }

      static toJson(src: NamedAmount) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, NamedAmount);


    let decoder = new ClassJsonDecoder();

    let jsonStr = '{"_class": "NamedPoint", "name": "hello", "x": 3, "y": 7}';

    let actual = decoder.decode(jsonStr);

    this.assertEqual('class name', 'NamedPoint', actual.constructor.name);
    this.assertEqual('name', 'hello', actual.name);
    this.assertEqual('x', 3, actual.x);
    this.assertEqual('y', 7, actual.y);


    jsonStr = '{"_class": "NamedAmount", "name": "hello", "amount": 42}';

    actual = decoder.decode(jsonStr);

    this.assertEqual('class name', 'NamedAmount', actual.constructor.name);
    this.assertEqual('name', 'hello', actual.name);
    this.assertEqual('amount', 42, actual.amount);


    jsonStr = '{"_class": "NamedThing", "name": "hello"}';

    try
    {
      actual = decoder.decode(jsonStr);
      this.fail("No exception for decoding abstract class.");
    }
    catch (ex)
    {
      this.assertEqual("abstract message",
                       true,
                       ex.message.indexOf("abstract class") > 0);
    }
  }

  /** Test that links are handled correctly.
   */
  testLinks() : void
  {
    // A band is a list of musicians and instruments.
    // Every musician has an instrument.

    let desc: ClassJsonDesc = new ClassJsonDesc("TestClassInstrument");
    desc.addField('id', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassInstrument
    {
      id: number;

      constructor(id: number)
      {
        this.id = id;
      }

      static fromJson(src: ObjectMap<any>) : TestClassInstrument
      {
        return new TestClassInstrument(<number>src['id']);
      }

      static toJson(src: TestClassInstrument) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassInstrument);

    desc = new ClassJsonDesc("TestClassMusician");
    desc.addField('id', new JsonPrimitiveType('int'));
    desc.addField('instrument', new JsonLink("TestClassInstrument",
                                             "../instruments",
                                             ["id"]));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassMusician
    {
      id: number;
      instrument: TestClassInstrument;
      constructor(id: number, instrument: TestClassInstrument)
      {
        this.id = id;
        this.instrument = instrument;
      }

      static fromJson(src: ObjectMap<any>) : TestClassMusician
      {
        return new TestClassMusician(<number>src['id'],
                                     <TestClassInstrument>src['instrument']);
      }

      static toJson(src: TestClassMusician) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassMusician);

    desc = new ClassJsonDesc("TestClassBand");
    desc.addField('musicians',
                  new JsonList(new JsonObj("TestClassMusician")));
    desc.addField('instruments',
                  new JsonList(new JsonObj("TestClassInstrument")));

    class TestClassBand
    {
      musicians: Array<TestClassMusician>;
      instruments: Array<TestClassInstrument>;
      constructor(theMusicians: Array<TestClassMusician>,
                  theInstruments: Array<TestClassInstrument>)
      {
        this.musicians = theMusicians;
        this.instruments = theInstruments;
      }

      static fromJson(src: ObjectMap<any>) : TestClassBand
      {
        return new TestClassBand(
            <Array<TestClassMusician>>src['musicians'],
            <Array<TestClassInstrument>>src['instruments']
        );
      }

      static toJson(src: TestClassBand) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassBand);
    ClassJsonRegistry.registry.addDesc(desc);

    let instruments: Array<ObjectMap<any>> = [
      {'_class': 'TestClassInstrument', 'id': 20},
      {'_class': 'TestClassInstrument', 'id': 21},
      {'_class': 'TestClassInstrument', 'id': 22},
    ];

    let musicians: Array<ObjectMap<any>> = [
      {'_class': 'TestClassMusician', 'id': 10, 'instrument': [22]},
      {'_class': 'TestClassMusician', 'id': 11, 'instrument': [20]},
    ];

    let bandJson: ObjectMap<any> = {'_class': 'TestClassBand',
                                    'musicians': musicians,
                                    'instruments': instruments};

    let bandJsonStr = JSON.stringify(bandJson);
    let band = <TestClassBand>(new ClassJsonDecoder()).decode(bandJsonStr);

    this.assertEqual('instrument link 0 decoded',
                     true,
                     band.instruments[2] === band.musicians[0].instrument);
    this.assertEqual('instrument link 1 decoded',
                     true,
                     band.instruments[0] === band.musicians[1].instrument);
  }

  //------------------------------------------------------------------
  // Private Utility Methods
  //------------------------------------------------------------------

} // END class TestClassJsonDecoder

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestClassJsonDecoder());
