//------------------------------------------------------------------
//  TestJsonPathElement.ts
//  Copyright 2019 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from '../../../util/string';
import { UnitTest } from '../../../unittest/UnitTest';
import { UnitTestRunner } from '../../../unittest/UnitTestRunner';
import { JsonPathElement } from '../JsonPathElement';
//------------------------------------------------------------------

/** Unit test for the JsonPathElement class.
 */
export class TestJsonPathElement extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonPath object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------


  /** Test creating the object.
   */
  testCtor() : void
  {
    let element: JsonPathElement = new JsonPathElement("foo", 3);

    this.assertEqual("name", "foo", element.name);
    this.assertEqual("index", 3, element.index);

    // Check that no exception raised and raise coverage.
    element.toString();
  }

  /** Test parsing strings.
   */
  testParse() : void
  {
    let element = JsonPathElement.parse("foo[3]", "path/to/foo[3]");

    this.assertEqual("name", "foo", element.name);
    this.assertEqual("index", 3, element.index);

    element = JsonPathElement.parse("foo", "path/to/foo");

    this.assertEqual("name", "foo", element.name);
    this.assertEqual("index", null, element.index);


    // Reversed brackets.

    try
    {
      JsonPathElement.parse("foo]3[", "path/to/foo]3[");
      this.fail("No exception for reversed brackets.");
    }
    catch
    {
    }

    // Bad index.

    try
    {
      JsonPathElement.parse("foo[badIndex]", "path/to/foo[badIndex]");
      this.fail("No exception for bad index.");
    }
    catch
    {
    }
  }

  /** Test resolving.
   */
  testResolve() : void
  {
    class Child
    {
    }

    class Parent
    {
      theChild: Child;
      theChildren: Array<Child>;

      constructor(theChild: Child, theChildren: Array<Child>)
      {
        this.theChild = theChild;
        this.theChildren = theChildren;
      }
    }

    let child = new Child();
    let children = [new Child(), new Child(), new Child()];
    let parent = new Parent(child, children);

    let element = JsonPathElement.parse("theChild", "theChild");
    let resolved = element.resolve(parent);

    this.assertEqual("no index", child, resolved);

    element = JsonPathElement.parse("theChildren[1]", "theChildren[1]");
    resolved = element.resolve(parent);

    this.assertEqual("index 1", children[1], resolved);

    // Child is not a list.

    element = JsonPathElement.parse("theChild[1]", "theChild[1]");
    try
    {
      element.resolve(parent);
      this.fail("No exception for not a list.");
    }
    catch
    {
    }

    // List too short.

    element = JsonPathElement.parse("theChildren[4]", "theChildren[4]");
    try
    {
      element.resolve(parent);
      this.fail("No exception for list too short.");
    }
    catch
    {
    }
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestJsonPathElement

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestJsonPathElement());
