//------------------------------------------------------------------
//  TestJsonObj.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../../util/string";
import * as axeTypes from "../../../util/types";
import { ClassJsonDesc } from '../../ClassJsonDesc';
import { ClassJsonRegistry } from '../../ClassJsonRegistry';
import { Day } from "../../../date/Day";
import { JsonDate } from '../JsonDate';
import { JsonLink } from '../JsonLink';
import { JsonList } from '../JsonList';
import { JsonObj } from '../JsonObj';
import { JsonPrimitiveType } from '../JsonPrimitiveType';
import { JsonType } from '../JsonType';
import { ObjectMap } from "../../../util/types";
import { UnitTest } from '../../../unittest/UnitTest';
import { UnitTestRunner } from '../../../unittest/UnitTestRunner';
//------------------------------------------------------------------

/** Unit test for the JsonObj class.
 */
export class TestJsonObj extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonObj object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /** A wrong class type should throw an exception.
   */
  testBadClassType() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestClass");
    desc.addField('fooAttr', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass
    {
      fooAttr: number;

      constructor(fooAttr: number)
      {
        this.fooAttr = fooAttr;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(<number>src['fooAttr']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);


    desc = new ClassJsonDesc("InvalidClass");
    ClassJsonRegistry.registry.addDesc(desc);

    class InvalidClass
    {
      static fromJson(src: ObjectMap<any>) : InvalidClass
      {
        return new InvalidClass();
      }

      static toJson(src: InvalidClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, InvalidClass);


    let jsonType = new JsonObj("TestClass");

    let errorMsg = jsonType.validate(new InvalidClass());
    this.assertNotNull('not an object', errorMsg);

    errorMsg = jsonType.validateJson({'_class': 'InvalidClass'});
    this.assertNotNull('Wrong _class name.', errorMsg);

    errorMsg = jsonType.validateJson({'_class': 'NotRegisteredClass'});
    this.assertNotNull('Un-registered _class name.', errorMsg);
  }

  /** Test the validateJson() method.
   */
  testValidateJson() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestBaseClass");
    desc.setAbstract(true);
    ClassJsonRegistry.registry.addDesc(desc);

    class TestBaseClass
    {
      static fromJson(src: ObjectMap<any>) : TestBaseClass
      {
        return null;
      }

      static toJson(src: TestBaseClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestBaseClass);


    desc = new ClassJsonDesc("TestClass");
    desc.setBaseClass("TestBaseClass");
    desc.addField('fooAttr', new JsonPrimitiveType('int'));
    desc.addField('barAttr', new JsonDate());
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass extends TestBaseClass
    {
      fooAttr: number;
      barAttr: Day;

      constructor(fooAttr: number, barAttr: Day)
      {
        super();
        this.fooAttr = fooAttr;
        this.barAttr = barAttr;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(src['fooAttr'], src['barAttr']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);


    let jsonType: JsonType = new JsonObj('TestClass');
    let baseType: JsonType = new JsonObj('TestBaseClass');

    // Valid objects.

    let errorMsg = jsonType.validate(null);
    this.assertNull('null is valid', errorMsg);

    errorMsg = jsonType.validate(new TestClass(3, new Day(2016, 11, 1)));
    this.assertNull('valid', errorMsg);

    let jsonObj: ObjectMap<any> =
      {'_class': 'TestClass', 'fooAttr': 3, 'barAttr': '2016-11-01'};
    errorMsg = jsonType.validateJson(jsonObj);
    this.assertNull('valid JSON', errorMsg);

    errorMsg = baseType.validate(new TestClass(3, new Day(2016, 11, 1)));
    this.assertNull('base valid', errorMsg);

    jsonObj = {'_class': 'TestClass', 'fooAttr': 3, 'barAttr': '2016-11-01'};
    errorMsg = baseType.validateJson(jsonObj);
    this.assertNull('base valid JSON', errorMsg);

    // Abstract class.

    errorMsg = baseType.validate(new TestBaseClass());
    this.assertNotNull('abstract', errorMsg);

    errorMsg = baseType.validateJson({'_class': 'TestBaseClass'});
    this.assertNotNull('abstract JSON', errorMsg);

    // Missing _class member.

    errorMsg = jsonType.validateJson({});
    this.assertNotNull('missing _class attribute', errorMsg);

    // Missing field.

    let obj = new TestClass(3, new Day(2016, 11, 1));
    delete obj.fooAttr;

    errorMsg = jsonType.validate(obj);
    this.assertNotNull('missing field', errorMsg);

    errorMsg = jsonType.validateJson({'_class': 'TestClass'});
    this.assertNotNull('missing field JSON', errorMsg);

    // Extra field.

    jsonObj = {'_class': 'TestClass',
               'fooAttr': 3,
               'barAttr': '2016-11-04',
               'fozAttr': 4,
               'bazAttr': 4};
    errorMsg = jsonType.validateJson(jsonObj);
    this.assertNotNull('extra field', errorMsg);

    // Missing and extra field.

    jsonObj = {'_class': 'TestClass',
               'barAttr': '2016-11-04',
               'fozAttr': 4,
               'bazAttr': 4};
    errorMsg = jsonType.validateJson(jsonObj);
    this.assertNotNull('missing and extra field', errorMsg);

    // Bad field type.

    errorMsg = jsonType.validate(new TestClass(3, <Day><any>'invalid type'));
    this.assertNotNull('bad field type', errorMsg);

    jsonObj = {'_class': 'TestClass', 'fooAttr': 3.14, 'barAttr': 4};
    errorMsg = jsonType.validateJson(jsonObj);
    this.assertNotNull('bad field type', errorMsg);
  }

  /** Test encoding and decoding subclasses.
   */
  testSubclasses() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TimeRange");
    desc.addField('beginsOn', new JsonDate());
    desc.addField('endsOn', new JsonDate());
    ClassJsonRegistry.registry.addDesc(desc);

    class TimeRange
    {
      beginsOn: Day;
      endsOn: Day;

      constructor(beginsOn: Day, endsOn: Day)
      {
        this.beginsOn = beginsOn;
        this.endsOn = endsOn;
      }

      static fromJson(src: ObjectMap<any>) : TimeRange
      {
        return new TimeRange(src['beginsOn'], src['endsOn']);
      }

      static toJson(src: TimeRange) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TimeRange);


    desc = new ClassJsonDesc("Wheel");
    desc.addField('width', new JsonPrimitiveType('int'));
    desc.addField('validRange', new JsonObj('TimeRange'));
    ClassJsonRegistry.registry.addDesc(desc);

    class Wheel
    {
      width: number;
      validRange: TimeRange;

      constructor(width: number, validRange: TimeRange)
      {
        this.width = width;
        this.validRange = validRange;
      }

      static fromJson(src: ObjectMap<any>) : Wheel
      {
        return new Wheel(src['width'], src['validRange']);
      }

      static toJson(src: Wheel) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, Wheel);


    desc = new ClassJsonDesc("Vehicle");
    desc.setAbstract(true);
    ClassJsonRegistry.registry.addDesc(desc);

    // This class should be abstract, but cannot be until Angular 2.0.1.
    // If you're at a later version of Angular that 2.0.1, please
    // add 'abstract' to this class.
    class Vehicle
    {
      static fromJson(src: ObjectMap<any>) : Vehicle
      {
        return null;
      }

      static toJson(src: Vehicle) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, Vehicle);


    desc = new ClassJsonDesc("Unicycle");
    desc.setBaseClass("Vehicle");
    desc.addField('wheel', new JsonObj('Wheel'));
    ClassJsonRegistry.registry.addDesc(desc);

    class Unicycle extends Vehicle
    {
      wheel: Wheel;

      constructor(wheel: Wheel)
      {
        super();
        this.wheel = wheel;
      }

      static fromJson(src: ObjectMap<any>) : Unicycle
      {
        return new Unicycle(<Wheel>src['wheel']);
      }

      static toJson(src: Unicycle) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, Unicycle);


    desc = new ClassJsonDesc("Car");
    desc.setBaseClass("Vehicle");
    desc.addField('wheels', new JsonList(new JsonObj('Wheel')));
    ClassJsonRegistry.registry.addDesc(desc);

    class Car extends Vehicle
    {
      wheels: Array<Wheel>;

      constructor(wheels: Array<Wheel>)
      {
        super();
        this.wheels = wheels;
      }

      static fromJson(src: ObjectMap<any>) : Car
      {
        return new Car(<Array<Wheel>>src['wheels']);
      }

      static toJson(src: Car) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, Car);


    desc = new ClassJsonDesc("Garage");
    desc.addField('vehicles', new JsonList(new JsonObj('Vehicle')));
    ClassJsonRegistry.registry.addDesc(desc);

    class Garage
    {
      vehicles: Array<Vehicle>;

      constructor(vehicles: Array<Vehicle>)
      {
        this.vehicles = vehicles;
      }

      static fromJson(src: ObjectMap<any>) : Garage
      {
        return new Garage(<Array<Vehicle>>src['vehicles']);
      }

      static toJson(src: Garage) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, Garage);


    let date1 = new Day(2015, 4, 1);
    let date2 = new Day(2016, 4, 1);
    let dateStr1 = '2015-04-01';
    let dateStr2 = '2016-04-01';

    let unicycle = new Unicycle(new Wheel(3, new TimeRange(date1, date2)));
    let wheelsList = [new Wheel(3, new TimeRange(date1, date2)),
                      new Wheel(3, new TimeRange(date1, date2))];
    let car = new Car(wheelsList);

    let garage = new Garage([unicycle, car]);


    let unicycleJson = {
      '_class': 'Unicycle',
      'wheel': {
        '_class': 'Wheel',
        'width': 3,
        'validRange': {
          '_class': 'TimeRange',
          'beginsOn': dateStr1,
          'endsOn': dateStr2
        }
      }
    };
    let wheelsJson = [
      {
        '_class': 'Wheel',
        'width': 3,
        'validRange': {
          '_class': 'TimeRange',
          'beginsOn': dateStr1,
          'endsOn': dateStr2
        }
      },
      {
        '_class': 'Wheel',
        'width': 3,
        'validRange': {
          '_class': 'TimeRange',
          'beginsOn': dateStr1,
          'endsOn': dateStr2
        }
      },
    ];

    let carJson = {
      '_class': 'Car',
      'wheels': wheelsJson,
    };

    let garageJson = {
      '_class': 'Garage',
      'vehicles': [
        unicycleJson,
        carJson
      ]
    };

    let jsonType = new JsonObj('Garage');


    // Test Encoding.

    this.assertEqual('encode null', null, jsonType.encode(null));
    this.assertEqual('encode', garageJson, jsonType.encode(garage));

    // Test Decoding.

    this.assertEqual('decode null', null, jsonType.decode(null));

    let actual = jsonType.decode(garageJson);

    this.assertEqual('decode # vehicles', 2, actual.vehicles.length);
    this.assertEqual('decode uni width', 3, actual.vehicles[0].wheel.width);
    this.assertEqual('decode # wheels', 2, actual.vehicles[1].wheels.length);
  }

  /** Test the encode() method.
   */
  testEncode() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestClass");
    desc.addField('fooAttr', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass
    {
      private fooAttr: number;

      constructor(fooAttr: number)
      {
        this.fooAttr = fooAttr;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(<number>src['fooAttr']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);


    let jsonType = new JsonObj('TestClass');
    let obj = new TestClass(3);
    let jsonObj = {'_class': 'TestClass', 'fooAttr': 3};
    this.assertEqual('encode', jsonObj, jsonType.encode(obj));
  }

  /** Test the decode() method.
   */
  testDecode() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestClass");
    desc.addField('fooAttr', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass
    {
      fooAttr: number;

      constructor(fooAttr: number)
      {
        this.fooAttr = fooAttr;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(<number>src['fooAttr']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);


    let jsonType = new JsonObj('TestClass');
    let obj = new TestClass(3);
    let jsonObj = {'_class': 'TestClass', 'fooAttr': 3};
    this.assertEqual('encode', jsonObj, jsonType.encode(obj));

    let actual: TestClass = <TestClass>jsonType.decode(jsonObj);
    this.assertEqual('decode', 3, actual.fooAttr);
  }

  /** Test the decodeLinks() method.
   */
  testDecodeLinks() : void
  {
    // A band is a list of musicians and instruments.
    // Every musician has an instrument.

    let desc: ClassJsonDesc = new ClassJsonDesc("TestClassInstrument");
    desc.addField('id', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassInstrument
    {
      id: number;

      constructor(id: number)
      {
        this.id = id;
      }

      static fromJson(src: ObjectMap<any>) : TestClassInstrument
      {
        return new TestClassInstrument(<number>src['id']);
      }

      static toJson(src: TestClassInstrument) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassInstrument);

    desc = new ClassJsonDesc("TestClassMusician");
    desc.addField('id', new JsonPrimitiveType('int'));
    desc.addField('instrument', new JsonLink("TestClassInstrument",
                                             "../instruments",
                                             ["id"]));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassMusician
    {
      id: number;
      instrument: TestClassInstrument;
      constructor(id: number, instrument: TestClassInstrument)
      {
        this.id = id;
        this.instrument = instrument;
      }

      static fromJson(src: ObjectMap<any>) : TestClassMusician
      {
        return new TestClassMusician(<number>src['id'],
                                     <TestClassInstrument>src['instrument']);
      }

      static toJson(src: TestClassMusician) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassMusician);

    desc = new ClassJsonDesc("TestClassBand");
    desc.addField('musicians',
                  new JsonList(new JsonObj("TestClassMusician")));
    desc.addField('instruments',
                  new JsonList(new JsonObj("TestClassInstument")));

    class TestClassBand
    {
      musicians: Array<TestClassMusician>;
      instruments: Array<TestClassInstrument>;
      constructor(theMusicians: Array<TestClassMusician>,
                  theInstruments: Array<TestClassInstrument>)
      {
        this.musicians = theMusicians;
        this.instruments = theInstruments;
      }

      static fromJson(src: ObjectMap<any>) : TestClassBand
      {
        return new TestClassBand(
            <Array<TestClassMusician>>src['musicians'],
            <Array<TestClassInstrument>>src['instruments']
        );
      }

      static toJson(src: TestClassBand) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassBand);

    let instruments = [
      new TestClassInstrument(20),
      new TestClassInstrument(21),
      new TestClassInstrument(22),
    ];

    let musicians = [
      new TestClassMusician(10, instruments[2]),
      new TestClassMusician(11, instruments[0]),
    ];

    let band = new TestClassBand(musicians, instruments);

    // Link are unresolved, so instead of pointers to objects,
    // they are the object IDs.

    for (let musician of musicians)
    {
      (<any>musician.instrument) = [musician.instrument.id];
    }

    let jsonType = new JsonObj('TestClassMusician');
    let parents: Array<object> = [band];

    this.assertNull('null', jsonType.decodeLinks(parents, null));

    jsonType.decodeLinks(parents, musicians[0]);

    this.assertEqual('musician 0', instruments[2], musicians[0].instrument);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestJsonObj

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestJsonObj());
