//------------------------------------------------------------------
//  JsonPrimitiveType.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import { Map } from '../../util/Map';
import * as axeString from "../../util/string";
import * as axeClasses from "../../util/classes";
import { JsonType } from './JsonType';
import { JsonTypeError } from './JsonTypeError';
//------------------------------------------------------------------

/** Marks a class attribute as an object of a primitive type.
 *
 * Types are: bool, float, int, string.
 */
export class JsonPrimitiveType extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** For each ClassJson type, the equivalent javascript type name.
   */
  static typeNameClasses: Map<string, string> = Map.asMap({
    'bool': 'boolean',
    'float': 'number',
    'int': 'number',
    'string': 'string'
  });

  /** The type that this object will encode/decode.
   */
  typeName: string;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonPrimitiveType object.
   */
  constructor(typeName: string)
  {
    super();

    this.typeName = typeName;

    let typeNameClasses = JsonPrimitiveType.typeNameClasses;
    if (!typeNameClasses.hasKey(typeName))
    {
      let msg = 'Invalid type name "' + typeName + '". Valid names are:';
      msg += typeNameClasses.keys();

      throw new Error(msg);
    }
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Checks that the specified value can be converted to JSON.
   *
   * @param value The value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validate(value: any) : JsonTypeError
  {
    let typeNameClasses = JsonPrimitiveType.typeNameClasses;

    if (value === null)
    {
      return null;
    }
    else if (typeof value != typeNameClasses.get(this.typeName))
    {
      let msg = ('is of type ' + this.typeName +
                 ' but the value is of type ' + axeClasses.className(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }

    // Enforce that a number of type 'int' is an integer.
    else if (this.typeName == 'int' &&
             !(isFinite(value) && Math.floor(value) == value))
    {
      let msg = ('is of type integer, but the value has a fraction.  ' +
                 'Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }
    else
    {
      return null;
    }
  }

  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    return this.validate(value);
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    return value;
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    return value;
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    return value;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      'typeName'
    ];
    return axeString.formatObject("JsonPrimitiveType", this, propertyNames);
  }

} // END class JsonPrimitiveType
