//------------------------------------------------------------------
//  TestMap.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeFunc from "../../util/func";
import * as axeString from "../../util/string";
import { IntMapType } from "../../util/Map";
import { ObjectMap } from "../../util/types";
import { Map } from "../../util/Map";
import { UnitTest } from "../../unittest/UnitTest";
import { UnitTestRunner } from "../../unittest/UnitTestRunner";
//------------------------------------------------------------------

/** Unit test for the Map class.
 */
export class TestMap extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new Map object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /* Test basic map functionality.
   */
  testMapBasic() : void
  {
    let map: Map<number, string> = Map.createIntMap<string>();

    this.assertEqual("empty size", 0, map.size());

    map.put(3, "3 value");
    map.put(5, "5 value");
    map.put(4, "4 value");

    this.assertEqual("size", 3, map.size());

    this.assertEqual("has key 2", false, map.hasKey(2));
    this.assertEqual("has key 3", true, map.hasKey(3));

    this.assertEqual("get 3", "3 value", map.get(3));
    this.assertEqual("get 4", "4 value", map.get(4));
    this.assertEqual("get 5", "5 value", map.get(5));

    // Test that keys come back in the correct order.

    let expectedKeys: Array<number> = [ 3, 5, 4 ];
    let actualKeys: Array<number> = map.keys();
    this.assertEqual("keys", expectedKeys, actualKeys);

    // Test that values come back in the correct order.

    let expectedValues = [ "3 value", "5 value", "4 value" ];
    let actualValues = map.values();
    this.assertEqual("values", expectedValues, actualValues);

    // Test that items come back in the correct order.

    let expectedItems = [{"key": 3, "value": "3 value"},
                         {"key": 4, "value": "4 value"},
                         {"key": 5, "value": "5 value"}];
    let actualItems = map.items();
    this.assertEqual("items", expectedItems, actualItems);

    // Test replacing a value.

    map.put(5, "7 value");
    this.assertEqual("after replace value", "7 value", map.get(5));
    this.assertEqual("after replace size", 3, map.size());

    // Test that toString() doesn't throw an exception.
    map.toString();
  }

  /* Test the MapSetterValue proxy.
   */
  testMapProxy() : void
  {
    let map = Map.createIntMap<string>();

    this.assertEqual("size", 0, map.size());

    map.put(3, "3 value");
    map.put(5, "5 value");
    map.put(4, "4 value");

    // Test getting.

    this.assertEqual("get", "3 value", map.access(3).value);
    this.assertEqual("get", "4 value", map.access(4).value);
    this.assertEqual("get", "5 value", map.access(5).value);

    // Test putting.

    map.access(6).value = "6 value";
    map.access(7).value = "7 value";
  }

  /* Test the putDefault() method.
   */
  testPutDefault() : void
  {
    let map = Map.createIntMap<string>();

    // Test setting a value.

    this.assertEqual("hasKey", false, map.hasKey(3));
    this.assertEqual("get", "3 value", map.putDefault(3, "3 value"));

    // Test not setting a value.
    //
    // We try to set 'new value', but it should *not* be set.

    this.assertEqual("hasKey", true, map.hasKey(3));
    this.assertEqual("get", "3 value", map.putDefault(3, "new value"));
  }

  /* Test the getDefault() method.
   */
  testGetDefault() : void
  {
    let map = Map.createIntMap<string>();

    map.put(3, "3 value");

    // Test getting a default.

    this.assertEqual("hasKey", false, map.hasKey(4));
    this.assertEqual("get", "4 default", map.getDefault(4, "4 default"));

    // Test getting a value.

    this.assertEqual("hasKey", true, map.hasKey(3));
    this.assertEqual("default", "3 value", map.getDefault(3, "3 default"));
  }

  /** Test converting to an object.
   */
  testObject() : void
  {
    let map: Map<string, number> = Map.createStringMap<number>();
    map.put('a', 1);
    map.put('b', 2);
    map.put('c', 3);

    let expected: ObjectMap<number> = {
      'a': 1,
      'b': 2,
      'c': 3,
    };

    let actual: ObjectMap<number> = map.toObject();

    this.assertEqual("toObject", expected, actual);

    // A non-string key type causes an exception.

    let map2: Map<number, number> = Map.createIntMap<number>();
    this.assertException("No exception for int key Object conversion.",
                         axeFunc.bind(map2, "toObject"));
  }

  /** Tests converting to/from JSON.
   */
  testMapJson() : void
  {
    let map: Map<number, number> = Map.createIntMap<number>();
    map.put(20, 1);
    map.put(18, 2);
    map.put(19, 3);

    let expected: Array<[number, number]> =
      [[20, 1],
       [18, 2],
       [19, 3]];
    let actual: Array<[number, number]> = map.toArrays();

    this.assertEqual("toArrays", expected, actual);

    map = Map.fromArrays<number, number>(actual, new IntMapType());

    let expectedItems = [{"key": 20, "value": 1},
                         {"key": 18, "value": 2},
                         {"key": 19, "value": 3}];
    let actualItems = map.items();
    this.assertEqual("fromArrays", expectedItems, actualItems);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestMap

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestMap());
