//------------------------------------------------------------------
//  array.js
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

import { Day } from '../date/Day';

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Returns true if the array contains the target object.
 */
export function contains<T>(theArray: Array<T>, target: T) : boolean
{
  return theArray.indexOf(target) != -1;
}

/** Returns the last object in the array.
 */
export function last<T>(theArray: Array<T>) : T
{
  if (theArray.length == 0)
  {
    throw Error("Cannot get last item of empty array.");
  }

  return theArray[theArray.length - 1];
}

/** Returns the index of the object in the array, or -1 if not present.
 */
export function indexOf<T>(theArray: Array<T>, target: T) : number
{
  return theArray.indexOf(target);
}

/** Returns the last index of the object in the array, or -1 if not present.
 */
export function lastIndexOf<T>(theArray: Array<T>, target: T) : number
{
  return theArray.lastIndexOf(target);
}

/** Returns the index of the last item that is less than or equal to the target.
 *
 * For example:
 *
 *   lastLessThanIndex([21, 22, 23, 24], 23.5)
 *
 * returns 2.
 *
 * @param theArray The array to be searched.
 * @param targetValue A value to be searched for.
 *
 * @return An index integer, or -1 if the target value is less than all items.
 */
export function lastLessThanIndex<T>(theArray: Array<T>,
                                     targetValue: T) : number
{
  for (let i = 0; i < theArray.length; i++)
  {
    let item = theArray[i];
    if (targetValue < item)
    {
      return i - 1;
    }
  }

  return theArray.length - 1;
}

/** Returns the index of the first item that is greater than or equal to target.
 *
 * For example:
 *
 *   firstGreaterThanIndex([21, 22, 23, 24], 23.5)
 *
 * returns 3.
 *
 * @param theArray The array to be searched.
 * @param targetValue A value to be searched for.
 *
 * @return An index integer, or -1 if the target value is greater
 *         than all items.
 */
export function firstGreaterThanIndex<T>(theArray: Array<T>,
                                         targetValue: T) : number
{
  for (let i = 0; i < theArray.length; i++)
  {
    let item = theArray[i];
    if (targetValue <= item)
    {
      return i;
    }
  }

  return -1;
}

/** Returns an array that is the specified array reversed.
 *
 * @param theArray The array to reverse.
 *
 * @return A newly created array that contains all the elements from the source
 *         array reversed.
 */
export function reversed<T>(theArray: Array<T>) : Array<T>
{
  let ret: Array<T> = createCopy(theArray);
  ret.reverse();
  return ret;
}

/** Inserts an item at a specified index.
 *
 * @param index The index at which the item will be found after the insert.
 * @param item The item to add.
 */
export function insert<T>(theArray: Array<T>, index: number, item: T) : void
{
  theArray.splice(index, 0, item);
}

/** Removes an item at a specified index.
 *
 * @param index The index of the item to remove.
 *
 * @return The removed item.
 */
export function removeAt<T>(theArray: Array<T>, index: number) : T
{
  let removedArray = theArray.splice(index, 1);
  return removedArray[0];
}

/** Removes the first instance of the target object from the array.
 *
 * @return True if the object was removed, false if the object was
 *         not in the array.
 */
export function remove<T>(theArray: Array<T>, target: T) : boolean
{
  let index = theArray.indexOf(target);
  if (index < 0)
  {
    return false;
  }
  else
  {
    theArray.splice(index, 1);
    return true;
  }
}

/** Removes all items from this array.
 */
export function removeAll(theArray: Array<any>) : void
{
  theArray.splice(0, theArray.length);
}

/** Removes all items from destArray, then adds all items from srcArray.
 *
 * The srcArray is not changed.
 */
export function replaceAll<T>(destArray: Array<T>, srcArray: Array<T>) : void
{
  removeAll(destArray);
  pushAll(srcArray, destArray);
}

/** Returns a new copy of the array with any duplicates removed.
 */
export function unique<T>(theArray: Array<T>) : Array<T>
{
  let newArray: Array<T> = [];

  for (let i = 0; i < theArray.length; ++i)
  {
    if (!contains(newArray, theArray[i]))
    {
      newArray.push(theArray[i]);
    }
  }
  return newArray;
}

/** Creates a new shallow copy of the specified array.
 */
export function createCopy<T>(srcArray: Array<T>) : Array<T>
{
  return srcArray.slice();
}

/** Copies all items from the source array to the destination array.
 *
 * This is a shallow copy.
 */
export function pushAll<T>(srcArray: Array<T>, destArray: Array<T>) : void
{
  // Instead of doing many calls to push() with a single argument,
  // we'll use apply() to call it like so:  push(item1, item2, item3...)
  //
  // Unfortunately, there are browser limitations to the length of the array
  // you're allowed to pass into apply(), so we will do the push()
  // in chunks of 16K items at a time.
  let chunkSize = 16000;

  for (let i = 0; i < srcArray.length; i += chunkSize)
  {
    let chunkArray = srcArray.slice(i, i + chunkSize);
    destArray.push.apply(destArray, chunkArray);
  }
}

/** Zips 2 arrays like the Python 'zip' function.
 *
 * Calling:   zip([1, 2, 3], [4, 5, 6])
 *
 * will return:  [[1, 4], [2, 5], [3, 6]]
 */
export function zip(...arrays: Array<Array<any>>) : Array<Array<any>>
{
  let numArrays: number = arrays.length;

  // Special case:  nothing passed in.

  if (numArrays == 0)
  {
    return [];
  }

  // Find the shortest array length.

  let arrayLength = arrays[0].length;

  for (let i = 0; i < numArrays; ++i)
  {
    if (arrays[i].length < arrayLength)
    {
      arrayLength = arrays[i].length;
    }
  }

  // Create the array to return.

  let ret: Array<Array<any>> = [];
  for (let i = 0; i < arrayLength; ++i)
  {
    ret.push([]);
  }

  // Populate the array.

  for (let i = 0; i < numArrays; ++i)
  {
    for (let j = 0; j < arrayLength; ++j)
    {
      ret[j].push(arrays[i][j]);
    }
  }

  return ret;
}

/** Returns the minimum value in the array.
 *
 * @param theArray The array to find the minimum value of.
 * @param accessor A function to apply to each item before comparing.
 */
export function min<T>(theArray: Array<T>,
                       accessor: (x: T) => (Date|number)) : T
{
  if (theArray.length == 0)
  {
    throw new Error("No min for empty array.");
  }

  // If no accessor was supplied, use a pass-through function.
  if (!accessor)
  {
    accessor = (x: T) => <Date|number><any>x;
  }

  // The minimum value encountered so far.
  let minValue: T = theArray[0];

  // The primitive version of the minimum, to be used in comparisons.
  let minPrimitive: number = accessor(minValue).valueOf();

  for (let i = 1; i < theArray.length; ++i)
  {
    let value: T = theArray[i];
    let primitive: number = accessor(value).valueOf();

    if (primitive < minPrimitive)
    {
      minValue = value;
      primitive = minPrimitive;
    }
  }

  return minValue;
}

/** Returns the maximum value in the array.
 *
 * @param theArray The array to find the minimum value of.
 * @param accessor A function to apply to each item before comparing.
 */
export function max<T>(theArray: Array<T>,
                       accessor: (x: T) => (Date|number)) : T
{
  if (theArray.length == 0)
  {
    throw new Error("No max for empty array.");
  }

  // If no accessor was supplied, use a pass-through function.
  if (!accessor)
  {
    accessor = (x: T) => <Date|number><any>x;
  }

  // The maximum value encountered so far.
  let maxValue: T = theArray[0];

  // The primitive version of the maximum, to be used in comparisons.
  let maxPrimitive: number = accessor(maxValue).valueOf();

  for (let i = 1; i < theArray.length; ++i)
  {
    let value: T = theArray[i];
    let primitive: number = accessor(value).valueOf();

    if (primitive > maxPrimitive)
    {
      maxValue = value;
      primitive = maxPrimitive;
    }
  }

  return maxValue;
}

/** Returns the mean of the specified array.
 *
 * Behavior is undefined if the array contains items that aren't numbers.
 */
export function mean(theArray: Array<number>) : number
{
  let total: number = 0;

  for (let i = 0; i < theArray.length; ++i) {
    total += theArray[i];
  }

  return total / theArray.length;
}

/** Helper function for 'axe.array.median'.  Do not use.
 *
 *  Will sort numbers ascending.
 */
export function compareNumbers(x: number, y: number) : number
{
  return x - y;
}

/** Returns the median of the specified array.
 *
 * Behavior is undefined if the array contains items that aren't numbers.
 */
export function median(theArray: Array<number>) : number
{
  if (theArray.length == 0) {
    return 0;
  }

  let sortedArray: Array<number> = theArray.slice(0);
  sortedArray.sort(compareNumbers);

  let mid: number = Math.floor(sortedArray.length / 2) - 1;
  if (mid < 0) {
    mid = 0;
  }

  return sortedArray[mid];
}

/** "Subtracts" one array from another.
 *
 * The formal definition is given in the description of the return value.
 *
 * Here is an example:
 *   subtract(["A", "B", "C", "D", "E"], ["A", "Squid", "D"])
 * should give
 *   ["B", "C", "E"]
 *
 * @param a An array.
 * @param b Another array.
 *
 * @return An array consisting of those elements alpha in a such that, for all
 *         elements beta in b, it is not true that alpha == beta. They will
 *         maintain their current ordering.
 */
export function subtract<T>(a: Array<T>, b: Array<T>) : Array<T>
{
  let newArray: Array<T> = [];

  for (let i = 0; i < a.length; i++)
  {
    if (!contains(b, a[i]))
    {
      newArray.push(a[i]);
    }
  }

  return newArray;
}

/** Tests two arrays for equality.
 *
 * Two arrays are equal if they are both null, or if they are both non-null,
 * have the same length, and member of the array at the same index either is
 * == to the other (if not an array) or axe.array.equals() (if an array).
 *
 * If both arguments are not arrays, instead returns a == b.
 */
export function equals(a: any, b: any) : boolean
{
  // Both nulls means equal.
  if (a == null && b == null)
  {
    return true;
  }

  // One null, the other not, means unequal.
  if ((a == null && b != null) || (a != null && b == null))
  {
    return false;
  }

  // Both Dates:  use valueOf().
  if (a instanceof Date && b instanceof Date)
  {
    return (a.valueOf() == b.valueOf());
  }

  // Both Days:  use valueOf().
  if (a instanceof Day && b instanceof Day)
  {
    return (a.valueOf() == b.valueOf());
  }

  // Neither arrays: use normal comparator.
  if (!(a instanceof Array) && !(b instanceof Array))
  {
    return (a == b);
  }

  // One is an array: not equal. (Normal == comparator returns [2] == 2,
  // bizarrely.)
  if (!(a instanceof Array) || !(b instanceof Array))
  {
    return false;
  }

  // Arrays of different lengths are unequal.
  if (a.length != b.length)
  {
    return false;
  }

  // If any index does not compare equal, the inputs are unequal.
  for (let i = 0; i < a.length; i++)
  {
    if (!equals(a[i], b[i]))
    {
      return false;
    }
  }

  return true;
}

/** Test that an object is an array.
 */
export function isArray(obj: any) : boolean
{
  return Object.prototype.toString.call(obj) == "[object Array]";
}
