//------------------------------------------------------------------
//  JsonPathElement.ts
//  Copyright 2019 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeArray from '../../util/array';
import * as axeClasses from "../../util/classes";
import * as axeString from '../../util/string';
//------------------------------------------------------------------

/** A single element of path spec.
 *
 * For example, in path 'foo/bar[3]', 'foo' and 'bar[3]' are two elements.
 */
export class JsonPathElement
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The name of the element.
   */
  name: string;

  /** If the element is a list, the index of the list item to choose.
   */
  index: number;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonPathElement object.
   */
  constructor(name: string, index: number)
  {
    this.name = name;
    this.index = index;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------


  /** Parses a string into a JsonPathElement object.
   *
   * @param elementStr The string to be parsed.
   * @param wholePath The whole path which the element is a part of.
   *                  To be used in error messages.
   *
   * @return A newly created JsonPathElement object.
   */
  static parse(elementStr: string, wholePath: string) : JsonPathElement
  {
    let indexLeft = elementStr.indexOf('[');
    let indexRight = elementStr.indexOf(']');

    if ((indexLeft < 0 && indexRight >= 0) ||
        (indexLeft >= 0 && indexRight < 0))
    {
      let msg = "Unbalanced brackets [] in element '" + elementStr + "'.  " +
        "Path: " + wholePath;
      throw new Error(msg);
    }

    if (indexLeft >= 0)
    {
      if (indexRight < indexLeft)
      {
        let msg = "Brackets [] are reversed in element '" + elementStr + "'." +
          "  Path: " + wholePath;
        throw new Error(msg);
      }

      let name = elementStr.substring(0, indexLeft);
      let indexStr = elementStr.substring(indexLeft + 1, indexRight);

      let index = parseInt(indexStr);
      if (isNaN(index))
      {
        let msg = "Index '" + indexStr + "' could not be parsed as a string " +
          "in element '" + elementStr + "'.  Path: " + wholePath;
        throw new Error(msg);
      }

      return new JsonPathElement(name, index);
    }
    else
    {
      return new JsonPathElement(elementStr, null);
    }
  }

  /** Resolves this path against the specified object.
   *
   * @param parentObject The object to return a child of.
   *
   * @return The child object.
   */
  resolve(parentObject: object) : object
  {
    let child: object = (<any>parentObject)[this.name];

    if (this.index !== null)
    {
      if (!(child instanceof Array))
      {
        let msg = "JsonPathElement " + this.pathString() + " expected to " +
          "find an array, but actually found a " + axeClasses.className(child) +
          ".  Value: " + child;
        throw new Error(msg);
      }

      if (child.length <= this.index)
      {
        let msg = "JsonPathElement " + this.pathString() + " expected to " +
          "find a list of " + this.index + " elements, but the list " +
          "actually contains " + child.length + ".  List: " + child;
        throw new Error(msg);
      }

      child = child[this.index];
    }

    return child;
  }

  /** Returns a label for display.
   */
  pathString() : string
  {
    if (this.index === null)
    {
      return this.name;
    }
    else
    {
      return this.name + '[' + this.index + ']';
    }
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      'name',
      'index'
    ];
    return axeString.formatObject("JsonPathElement", this, propertyNames);
  }

} // END class JsonPathElement
