//------------------------------------------------------------------
//  ClassJsonEncoder.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import { ClassJsonRegistry } from './ClassJsonRegistry';
import { JsonObj } from './jsonTypes/JsonObj';
import { JsonTypeError } from './jsonTypes/JsonTypeError';
import { className } from '../util/classes';
import * as axeString from "../util/string";
//------------------------------------------------------------------

/** Encodes an object into JSON.
 */
export class ClassJsonEncoder
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------


  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ClassJsonEncoder object.
   */
  constructor()
  {
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Encodes the specified object into JSON.
   *
   * @param obj The object to be encoded.
   * @param pretty True if the JSON should be formatted for human eyes.
   *
   * @return a JSON string.
   */
  encode(obj: any, pretty: boolean = false) : string
  {
    let clazz = obj.constructor;

    if (!ClassJsonRegistry.registry.isRegisteredClass(clazz))
    {
      let msg = "Attempting to JSON-encode an object that is not registered.\n";
      msg += "Class: " + className(obj) + "\n";
      msg += "Object: " + axeString.format(obj);
      throw new Error(msg);
    }

    let jsonDesc = ClassJsonRegistry.registry.getDescForClass(clazz);
    let jsonType = new JsonObj(jsonDesc.className);

    let errorMsg: JsonTypeError = jsonType.validate(obj);
    if (errorMsg)
    {
      let msg: string = errorMsg.format();
      msg += "\n\nUnable to encode object:\n" + axeString.format(obj);
      throw new Error(msg);
    }

    let jsonDict = jsonType.encode(obj);

    let numIndentSpaces: number = pretty ? 2 : 0;
    let jsonStr = JSON.stringify(jsonDict, null, numIndentSpaces);

    return jsonStr;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("ClassJsonEncoder", this, propertyNames);
  }

} // END class ClassJsonEncoder
